<?php

use App\Models\AppDefinition;
use App\Models\User;
use Illuminate\Support\Facades\Hash;

uses()->group('feature');

beforeEach(function () {
    $this->user = User::create([
        'username' => 'admin',
        'email' => 'admin@example.com',
        'password' => Hash::make('password'),
        'is_admin' => true,
    ]);
});

test('user can view library', function () {
    $response = $this->actingAs($this->user)->get('/admin/library');
    
    $response->assertStatus(200);
    $response->assertSee('App Library');
});

test('user can create app definition', function () {
    $response = $this->actingAs($this->user)->post('/admin/library', [
        'slug' => 'redis',
        'name' => 'Redis',
        'category' => 'database',
        'image' => 'redis',
        'tag' => 'latest',
        'restart_policy' => 'unless-stopped',
    ]);
    
    $response->assertRedirect();
    $this->assertDatabaseHas('app_definitions', [
        'slug' => 'redis',
        'name' => 'Redis',
    ]);
});

test('app definition slug must be unique', function () {
    AppDefinition::create([
        'slug' => 'unique-app',
        'name' => 'First',
        'image' => 'nginx',
    ]);
    
    $response = $this->actingAs($this->user)->post('/admin/library', [
        'slug' => 'unique-app',
        'name' => 'Second',
        'image' => 'nginx',
        'category' => 'other',
    ]);
    
    $response->assertSessionHasErrors('slug');
});

test('user can delete app definition', function () {
    $app = AppDefinition::create([
        'slug' => 'temp-app',
        'name' => 'Temp',
        'image' => 'nginx',
    ]);
    
    $response = $this->actingAs($this->user)->delete("/admin/library/{$app->id}");
    
    $response->assertRedirect();
    $this->assertDatabaseMissing('app_definitions', ['id' => $app->id]);
});
