<?php

use App\Models\AppDefinition;
use App\Models\Container;
use App\Models\User;
use Illuminate\Support\Facades\Hash;

uses()->group('feature');

beforeEach(function () {
    $this->user = User::create([
        'username' => 'admin',
        'email' => 'admin@example.com',
        'password' => Hash::make('password'),
        'is_admin' => true,
    ]);
    
    $this->appDef = AppDefinition::create([
        'slug' => 'nginx',
        'name' => 'Nginx',
        'image' => 'nginx',
        'tag' => 'latest',
        'ports' => ['8080' => '80'],
    ]);
});

test('user can view containers list', function () {
    $response = $this->actingAs($this->user)->get('/admin/containers');
    
    $response->assertStatus(200);
    $response->assertSee('Containers');
});

test('user can view container creation form', function () {
    $response = $this->actingAs($this->user)->get('/admin/containers/create');
    
    $response->assertStatus(200);
    $response->assertSee('Container Configuration');
});

test('user can create container record', function () {
    $response = $this->actingAs($this->user)->post('/admin/containers', [
        'name' => 'test-nginx',
        'app_definition_id' => $this->appDef->id,
        'domain' => '',
        'ssl_enabled' => false,
        'auto_start' => true,
    ]);
    
    $response->assertRedirect();
    $this->assertDatabaseHas('containers', [
        'name' => 'test-nginx',
        'app_definition_id' => $this->appDef->id,
    ]);
});

test('container name must be valid format', function () {
    $response = $this->actingAs($this->user)->post('/admin/containers', [
        'name' => 'Invalid Name With Spaces!',
        'app_definition_id' => $this->appDef->id,
    ]);
    
    $response->assertSessionHasErrors('name');
});

test('user can delete container', function () {
    $container = Container::create([
        'name' => 'delete-me',
        'app_definition_id' => $this->appDef->id,
        'status' => 'stopped',
    ]);
    
    $response = $this->actingAs($this->user)->delete("/admin/containers/{$container->id}");
    
    $response->assertRedirect();
    $this->assertDatabaseMissing('containers', ['id' => $container->id]);
});
