<?php

namespace App\Services;

use App\Models\Container;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Symfony\Component\Process\Process;

class NginxService
{
    protected string $sitesPath;
    protected string $enabledPath;
    protected string $nginxUser;
    protected string $baseName;

    public function __construct()
    {
        $this->sitesPath = config('panel.nginx.sites_path', '/etc/nginx/sites-available');
        $this->enabledPath = config('panel.nginx.enabled_path', '/etc/nginx/sites-enabled');
        $this->nginxUser = config('panel.nginx.user', 'www-data');
        $this->baseName = config('panel.base_name', 'homelab');
    }

    /**
     * Generate Nginx configuration for a container
     */
    public function generateConfig(Container $container): string
    {
        if (empty($container->domain)) {
            return '';
        }

        $serverName = $container->domain;
        $sslEnabled = $container->ssl_enabled;
        $containerName = $container->name;

        // Get target port from container
        $targetPort = $this->getTargetPort($container);

        if (!$targetPort) {
            Log::error("No target port found for container: {$container->name}");
            return '';
        }

        $config = $this->buildServerBlock($serverName, $containerName, $targetPort, $sslEnabled);

        return $config;
    }

    /**
     * Write Nginx configuration to disk
     */
    public function writeConfig(Container $container): bool
    {
        $config = $this->generateConfig($container);

        if (empty($config)) {
            return false;
        }

        $filename = $this->getConfigFilename($container);
        $filepath = $this->sitesPath . '/' . $filename;

        try {
            // Ensure sites-available directory exists
            if (!is_dir($this->sitesPath)) {
                mkdir($this->sitesPath, 0755, true);
            }

            // Write config
            file_put_contents($filepath, $config);
            
            // Set proper ownership
            chown($filepath, 'root');
            chgrp($filepath, 'root');
            chmod($filepath, 0644);

            // Enable site (create symlink)
            $this->enableSite($filename);

            Log::info("Nginx config written for container: {$container->name}");
            return true;
        } catch (\Exception $e) {
            Log::error("Failed to write Nginx config: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Remove Nginx configuration
     */
    public function removeConfig(Container $container): bool
    {
        $filename = $this->getConfigFilename($container);
        $filepath = $this->sitesPath . '/' . $filename;
        $enabledLink = $this->enabledPath . '/' . $filename;

        try {
            // Remove symlink first
            if (is_link($enabledLink) || file_exists($enabledLink)) {
                unlink($enabledLink);
            }

            // Remove config file
            if (file_exists($filepath)) {
                unlink($filepath);
            }

            Log::info("Nginx config removed for container: {$container->name}");
            return true;
        } catch (\Exception $e) {
            Log::error("Failed to remove Nginx config: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Test Nginx configuration
     */
    public function testConfig(): bool
    {
        $process = new Process(['nginx', '-t']);
        $process->run();
        return $process->isSuccessful();
    }

    /**
     * Reload Nginx
     */
    public function reload(): bool
    {
        if (!$this->testConfig()) {
            Log::error("Nginx config test failed, not reloading");
            return false;
        }

        $process = new Process(['systemctl', 'reload', 'nginx']);
        $process->run();
        return $process->isSuccessful();
    }

    /**
     * Build the server block configuration
     */
    protected function buildServerBlock(string $serverName, string $containerName, int $targetPort, bool $sslEnabled): string
    {
        $baseName = $this->baseName;
        
        if ($sslEnabled) {
            return $this->buildSslConfig($serverName, $containerName, $targetPort, $baseName);
        }

        return $this->buildHttpConfig($serverName, $containerName, $targetPort, $baseName);
    }

    /**
     * Build HTTP-only configuration
     */
    protected function buildHttpConfig(string $serverName, string $containerName, int $targetPort, string $baseName): string
    {
        $upstreamName = $this->sanitizeUpstreamName($containerName);

        return <<<NGINX
# Auto-generated by {$baseName}-panel for container: {$containerName}
upstream {$upstreamName} {
    server localhost:{$targetPort};
}

server {
    listen 80;
    listen [::]:80;
    server_name {$serverName};

    access_log /var/log/nginx/{$baseName}-{$containerName}-access.log;
    error_log /var/log/nginx/{$baseName}-{$containerName}-error.log;

    location / {
        proxy_pass http://{$upstreamName};
        proxy_http_version 1.1;
        proxy_set_header Upgrade \$http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host \$host;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;
        proxy_cache_bypass \$http_upgrade;
        proxy_read_timeout 86400;
    }
}
NGINX;
    }

    /**
     * Build SSL-enabled configuration
     */
    protected function buildSslConfig(string $serverName, string $containerName, int $targetPort, string $baseName): string
    {
        $upstreamName = $this->sanitizeUpstreamName($containerName);

        return <<<NGINX
# Auto-generated by {$baseName}-panel for container: {$containerName}
upstream {$upstreamName} {
    server localhost:{$targetPort};
}

# HTTP redirect to HTTPS
server {
    listen 80;
    listen [::]:80;
    server_name {$serverName};
    return 301 https://\$server_name\$request_uri;
}

server {
    listen 443 ssl http2;
    listen [::]:443 ssl http2;
    server_name {$serverName};

    ssl_certificate /etc/letsencrypt/live/{$serverName}/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/{$serverName}/privkey.pem;
    ssl_trusted_certificate /etc/letsencrypt/live/{$serverName}/chain.pem;

    # SSL configuration
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers HIGH:!aNULL:!MD5;
    ssl_prefer_server_ciphers on;
    ssl_session_cache shared:SSL:10m;
    ssl_session_timeout 10m;

    access_log /var/log/nginx/{$baseName}-{$containerName}-access.log;
    error_log /var/log/nginx/{$baseName}-{$containerName}-error.log;

    location / {
        proxy_pass http://{$upstreamName};
        proxy_http_version 1.1;
        proxy_set_header Upgrade \$http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host \$host;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;
        proxy_cache_bypass \$http_upgrade;
        proxy_read_timeout 86400;
    }
}
NGINX;
    }

    /**
     * Get the target port from container configuration
     */
    protected function getTargetPort(Container $container): ?int
    {
        $ports = $container->port_mappings ?? [];

        // Find the first host port
        foreach ($ports as $hostPort => $containerPort) {
            return (int) $hostPort;
        }

        return null;
    }

    /**
     * Get the config filename for a container
     */
    protected function getConfigFilename(Container $container): string
    {
        return $this->baseName . '-' . $container->name . '.conf';
    }

    /**
     * Enable a site by creating symlink
     */
    protected function enableSite(string $filename): bool
    {
        $source = $this->sitesPath . '/' . $filename;
        $target = $this->enabledPath . '/' . $filename;

        // Remove existing symlink if any
        if (is_link($target) || file_exists($target)) {
            unlink($target);
        }

        // Ensure enabled directory exists
        if (!is_dir($this->enabledPath)) {
            mkdir($this->enabledPath, 0755, true);
        }

        return symlink($source, $target);
    }

    /**
     * Sanitize container name for upstream
     */
    protected function sanitizeUpstreamName(string $name): string
    {
        return preg_replace('/[^a-zA-Z0-9_]/', '_', $this->baseName . '_' . $name);
    }

    /**
     * Check if Certbot certificate exists
     */
    public function certificateExists(string $domain): bool
    {
        $certPath = "/etc/letsencrypt/live/{$domain}/fullchain.pem";
        return file_exists($certPath);
    }

    /**
     * Request certificate via Certbot
     */
    public function requestCertificate(string $domain, string $email): bool
    {
        $process = new Process([
            'certbot',
            'certonly',
            '--nginx',
            '-d', $domain,
            '--email', $email,
            '--agree-tos',
            '--non-interactive',
            '--quiet'
        ]);

        $process->setTimeout(300);
        $process->run();

        return $process->isSuccessful();
    }

    /**
     * Setup automatic certificate renewal
     */
    public function setupAutoRenewal(): bool
    {
        // Certbot automatically installs a systemd timer or cron job
        // We just need to verify it's in place
        $process = new Process(['systemctl', 'is-active', 'certbot.timer']);
        $process->run();

        if (!$process->isSuccessful()) {
            // Try to enable the timer
            $process = new Process(['systemctl', 'enable', '--now', 'certbot.timer']);
            $process->run();
        }

        return true;
    }
}
