<?php

namespace App\Services;

use App\Models\Container;
use Illuminate\Support\Facades\Log;
use Symfony\Component\Process\Process;
use Symfony\Component\Process\Exception\ProcessFailedException;

class DockerService
{
    protected string $socket;
    protected string $composePath;
    protected string $network;

    public function __construct()
    {
        $this->socket = config('panel.docker.socket', '/var/run/docker.sock');
        $this->composePath = config('panel.docker.compose_path', '/usr/local/bin/docker-compose');
        $this->network = config('panel.docker.network', config('panel.base_name').'-network');
    }

    /**
     * Check if Docker is installed and running
     */
    public function isAvailable(): bool
    {
        try {
            $process = $this->runDockerCommand(['version']);
            return $process->isSuccessful();
        } catch (\Exception $e) {
            Log::error('Docker not available: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * List all containers
     */
    public function listContainers(bool $all = true): array
    {
        $cmd = ['ps', '--format', 'json'];
        if ($all) {
            array_unshift($cmd, '-a');
        }

        $process = $this->runDockerCommand($cmd);
        
        if (!$process->isSuccessful()) {
            return [];
        }

        $output = $process->getOutput();
        $containers = [];

        // Docker can output multiple JSON objects, one per line
        foreach (explode("\n", trim($output)) as $line) {
            if (empty($line)) continue;
            $data = json_decode($line, true);
            if ($data) {
                $containers[] = $data;
            }
        }

        return $containers;
    }

    /**
     * Get container details
     */
    public function inspectContainer(string $containerId): ?array
    {
        $process = $this->runDockerCommand(['inspect', $containerId]);

        if (!$process->isSuccessful()) {
            return null;
        }

        $output = json_decode($process->getOutput(), true);
        return $output[0] ?? null;
    }

    /**
     * Get container logs
     */
    public function getLogs(string $containerId, int $tail = 100): string
    {
        $process = $this->runDockerCommand([
            'logs',
            '--tail', (string) $tail,
            $containerId
        ]);

        return $process->isSuccessful() ? $process->getOutput() : '';
    }

    /**
     * Pull an image
     */
    public function pullImage(string $image): bool
    {
        $process = $this->runDockerCommand(['pull', $image]);
        return $process->isSuccessful();
    }

    /**
     * Run a new container
     */
    public function runContainer(array $config): ?string
    {
        $cmd = ['run', '-d'];

        // Add name
        if (isset($config['name'])) {
            $cmd[] = '--name';
            $cmd[] = $config['name'];
        }

        // Add network
        if (isset($config['network'])) {
            $cmd[] = '--network';
            $cmd[] = $config['network'];
        }

        // Add restart policy
        if (isset($config['restart'])) {
            $cmd[] = '--restart';
            $cmd[] = $config['restart'];
        }

        // Add environment variables
        if (isset($config['environment']) && is_array($config['environment'])) {
            foreach ($config['environment'] as $key => $value) {
                $cmd[] = '-e';
                $cmd[] = $key . '=' . $value;
            }
        }

        // Add port mappings
        if (isset($config['ports']) && is_array($config['ports'])) {
            foreach ($config['ports'] as $host => $container) {
                $cmd[] = '-p';
                $cmd[] = $host . ':' . $container;
            }
        }

        // Add volumes
        if (isset($config['volumes']) && is_array($config['volumes'])) {
            foreach ($config['volumes'] as $host => $container) {
                $cmd[] = '-v';
                $cmd[] = $host . ':' . $container;
            }
        }

        // Add labels
        if (isset($config['labels']) && is_array($config['labels'])) {
            foreach ($config['labels'] as $key => $value) {
                $cmd[] = '--label';
                $cmd[] = $key . '=' . $value;
            }
        }

        // Add the image
        $cmd[] = $config['image'];

        $process = $this->runDockerCommand($cmd);

        if (!$process->isSuccessful()) {
            Log::error('Failed to run container: ' . $process->getErrorOutput());
            return null;
        }

        return trim($process->getOutput());
    }

    /**
     * Start a container
     */
    public function startContainer(string $containerId): bool
    {
        $process = $this->runDockerCommand(['start', $containerId]);
        return $process->isSuccessful();
    }

    /**
     * Stop a container
     */
    public function stopContainer(string $containerId, int $timeout = 10): bool
    {
        $process = $this->runDockerCommand(['stop', '-t', (string) $timeout, $containerId]);
        return $process->isSuccessful();
    }

    /**
     * Restart a container
     */
    public function restartContainer(string $containerId, int $timeout = 10): bool
    {
        $process = $this->runDockerCommand(['restart', '-t', (string) $timeout, $containerId]);
        return $process->isSuccessful();
    }

    /**
     * Remove a container
     */
    public function removeContainer(string $containerId, bool $force = false, bool $volumes = false): bool
    {
        $cmd = ['rm'];
        
        if ($force) {
            $cmd[] = '-f';
        }
        
        if ($volumes) {
            $cmd[] = '-v';
        }
        
        $cmd[] = $containerId;

        $process = $this->runDockerCommand($cmd);
        return $process->isSuccessful();
    }

    /**
     * Get container stats
     */
    public function getStats(string $containerId): ?array
    {
        $process = $this->runDockerCommand([
            'stats',
            '--no-stream',
            '--format', 'json',
            $containerId
        ]);

        if (!$process->isSuccessful()) {
            return null;
        }

        return json_decode($process->getOutput(), true);
    }

    /**
     * Ensure the application network exists
     */
    public function ensureNetwork(): bool
    {
        // Check if network exists
        $process = $this->runDockerCommand(['network', 'ls', '--format', '{{.Name}}']);
        $networks = explode("\n", trim($process->getOutput()));

        if (in_array($this->network, $networks)) {
            return true;
        }

        // Create the network
        $process = $this->runDockerCommand([
            'network', 'create',
            '--driver', 'bridge',
            $this->network
        ]);

        return $process->isSuccessful();
    }

    /**
     * Get available port on host
     */
    public function findAvailablePort(int $start = 10000, int $end = 65535): ?int
    {
        // Get all used ports
        $process = $this->runDockerCommand([
            'ps', '-q'
        ]);

        $usedPorts = [];
        $containerIds = array_filter(explode("\n", trim($process->getOutput())));

        foreach ($containerIds as $id) {
            $inspect = $this->inspectContainer($id);
            if ($inspect && isset($inspect['NetworkSettings']['Ports'])) {
                foreach ($inspect['NetworkSettings']['Ports'] as $containerPort => $hostBindings) {
                    if (is_array($hostBindings)) {
                        foreach ($hostBindings as $binding) {
                            if (isset($binding['HostPort'])) {
                                $usedPorts[] = (int) $binding['HostPort'];
                            }
                        }
                    }
                }
            }
        }

        // Find first available port
        for ($port = $start; $port <= $end; $port++) {
            if (!in_array($port, $usedPorts)) {
                // Also check if port is in use by system
                $socket = @fsockopen('localhost', $port, $errno, $errstr, 0.1);
                if (!$socket) {
                    return $port;
                }
                fclose($socket);
            }
        }

        return null;
    }

    /**
     * Run a docker command
     */
    protected function runDockerCommand(array $args): Process
    {
        $cmd = array_merge(['docker'], $args);
        $process = new Process($cmd);
        $process->setTimeout(300);
        $process->run();

        return $process;
    }
}
