<?php

namespace App\Services;

use App\Models\AppDefinition;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Symfony\Component\Yaml\Yaml;

class ConfigService
{
    protected string $libraryPath;
    protected string $baseName;

    public function __construct()
    {
        $this->baseName = config('panel.base_name', 'homelab');
        $this->libraryPath = config('panel.library.path', '/etc/'.$this->baseName.'/library');
    }

    /**
     * Load all app definitions from library directory
     */
    public function loadLibrary(): array
    {
        $definitions = [];

        if (!is_dir($this->libraryPath)) {
            Log::warning("Library path does not exist: {$this->libraryPath}");
            return $definitions;
        }

        $files = File::files($this->libraryPath);

        foreach ($files as $file) {
            $extension = $file->getExtension();
            
            if (!in_array($extension, ['yml', 'yaml', 'json'])) {
                continue;
            }

            $definition = $this->parseFile($file->getRealPath());
            
            if ($definition) {
                $definitions[] = $definition;
            }
        }

        return $definitions;
    }

    /**
     * Parse a single definition file
     */
    public function parseFile(string $filepath): ?array
    {
        try {
            $content = file_get_contents($filepath);
            $extension = pathinfo($filepath, PATHINFO_EXTENSION);

            if (in_array($extension, ['yml', 'yaml'])) {
                $data = Yaml::parse($content);
            } elseif ($extension === 'json') {
                $data = json_decode($content, true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    throw new \Exception("JSON parse error: " . json_last_error_msg());
                }
            } else {
                return null;
            }

            // Validate required fields
            if (!$this->validateDefinition($data)) {
                Log::error("Invalid app definition in: {$filepath}");
                return null;
            }

            return $data;

        } catch (\Exception $e) {
            Log::error("Failed to parse definition file {$filepath}: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Save an app definition to file
     */
    public function saveDefinition(array $definition): bool
    {
        if (!$this->validateDefinition($definition)) {
            return false;
        }

        $slug = $definition['slug'];
        $filename = $slug . '.yml';
        $filepath = $this->libraryPath . '/' . $filename;

        try {
            // Ensure directory exists with proper permissions
            if (!is_dir($this->libraryPath)) {
                mkdir($this->libraryPath, 0750, true);
                chown($this->libraryPath, 'root');
                chgrp($this->libraryPath, $this->baseName);
            }

            $yaml = Yaml::dump($definition, 4, 2);
            file_put_contents($filepath, $yaml);

            // Set proper ownership
            chown($filepath, 'root');
            chgrp($filepath, $this->baseName);
            chmod($filepath, 0640);

            return true;
        } catch (\Exception $e) {
            Log::error("Failed to save definition: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Delete an app definition
     */
    public function deleteDefinition(string $slug): bool
    {
        $filename = $slug . '.yml';
        $filepath = $this->libraryPath . '/' . $filename;

        if (file_exists($filepath)) {
            return unlink($filepath);
        }

        return false;
    }

    /**
     * Get a single definition by slug
     */
    public function getDefinition(string $slug): ?array
    {
        $filepath = $this->libraryPath . '/' . $slug . '.yml';

        if (!file_exists($filepath)) {
            // Try JSON
            $filepath = $this->libraryPath . '/' . $slug . '.json';
            if (!file_exists($filepath)) {
                return null;
            }
        }

        return $this->parseFile($filepath);
    }

    /**
     * Validate app definition structure
     */
    protected function validateDefinition(array $data): bool
    {
        $required = ['slug', 'name', 'image'];

        foreach ($required as $field) {
            if (empty($data[$field])) {
                Log::error("Missing required field in app definition: {$field}");
                return false;
            }
        }

        // Validate slug format
        if (!preg_match('/^[a-z0-9-]+$/', $data['slug'])) {
            Log::error("Invalid slug format: {$data['slug']}");
            return false;
        }

        return true;
    }

    /**
     * Sync library to database
     */
    public function syncToDatabase(): int
    {
        $definitions = $this->loadLibrary();
        $count = 0;

        foreach ($definitions as $data) {
            AppDefinition::updateOrCreate(
                ['slug' => $data['slug']],
                [
                    'name' => $data['name'],
                    'description' => $data['description'] ?? '',
                    'category' => $data['category'] ?? 'other',
                    'image' => $data['image'],
                    'registry' => $data['registry'] ?? null,
                    'tag' => $data['tag'] ?? 'latest',
                    'ports' => $data['ports'] ?? [],
                    'environment_defaults' => $data['environment'] ?? [],
                    'volumes' => $data['volumes'] ?? [],
                    'labels' => $data['labels'] ?? [],
                    'network_mode' => $data['network_mode'] ?? null,
                    'restart_policy' => $data['restart_policy'] ?? 'unless-stopped',
                    'custom_image_allowed' => $data['custom_image_allowed'] ?? false,
                    'documentation_url' => $data['documentation_url'] ?? null,
                    'icon' => $data['icon'] ?? null,
                ]
            );
            $count++;
        }

        return $count;
    }

    /**
     * Create default library structure
     */
    public function createDefaultLibrary(): void
    {
        $defaults = [
            $this->getNginxProxyDefinition(),
            $this->getPortainerDefinition(),
            $this->getPiholeDefinition(),
        ];

        foreach ($defaults as $definition) {
            $this->saveDefinition($definition);
        }
    }

    /**
     * Get nginx-proxy definition
     */
    protected function getNginxProxyDefinition(): array
    {
        return [
            'slug' => 'nginx-proxy',
            'name' => 'Nginx Proxy Manager',
            'description' => 'Easy to use reverse proxy manager with SSL support',
            'category' => 'networking',
            'image' => 'jc21/nginx-proxy-manager',
            'tag' => 'latest',
            'ports' => [
                '81' => '81',
                '80' => '80',
                '443' => '443',
            ],
            'volumes' => [
                '/var/lib/'.$this->baseName.'/npm/data' => '/data',
                '/var/lib/'.$this->baseName.'/npm/letsencrypt' => '/etc/letsencrypt',
            ],
            'environment' => [
                'DB_SQLITE_FILE' => '/data/database.sqlite',
            ],
            'restart_policy' => 'unless-stopped',
        ];
    }

    /**
     * Get Portainer definition
     */
    protected function getPortainerDefinition(): array
    {
        return [
            'slug' => 'portainer',
            'name' => 'Portainer CE',
            'description' => 'Container management UI for Docker',
            'category' => 'management',
            'image' => 'portainer/portainer-ce',
            'tag' => 'latest',
            'ports' => [
                '9000' => '9000',
            ],
            'volumes' => [
                '/var/run/docker.sock' => '/var/run/docker.sock',
                '/var/lib/'.$this->baseName.'/portainer/data' => '/data',
            ],
            'restart_policy' => 'unless-stopped',
        ];
    }

    /**
     * Get Pi-hole definition
     */
    protected function getPiholeDefinition(): array
    {
        return [
            'slug' => 'pihole',
            'name' => 'Pi-hole',
            'description' => 'Network-wide ad blocking DNS server',
            'category' => 'networking',
            'image' => 'pihole/pihole',
            'tag' => 'latest',
            'ports' => [
                '8080' => '80',
                '53' => '53/tcp',
                '53' => '53/udp',
            ],
            'environment' => [
                'TZ' => 'UTC',
                'WEBPASSWORD' => '',
            ],
            'volumes' => [
                '/var/lib/'.$this->baseName.'/pihole/etc-pihole' => '/etc/pihole',
                '/var/lib/'.$this->baseName.'/pihole/etc-dnsmasq.d' => '/etc/dnsmasq.d',
            ],
            'restart_policy' => 'unless-stopped',
        ];
    }
}
