<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AppDefinition;
use App\Services\ConfigService;
use Illuminate\Http\Request;

class LibraryController extends Controller
{
    public function __construct(
        protected ConfigService $configService
    ) {}

    public function index()
    {
        $definitions = AppDefinition::all()->groupBy('category');
        return view('admin.library.index', compact('definitions'));
    }

    public function create()
    {
        return view('admin.library.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'slug' => 'required|string|regex:/^[a-z0-9-]+$/|unique:app_definitions',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category' => 'required|string|in:networking,management,media,development,other',
            'image' => 'required|string',
            'registry' => 'nullable|string',
            'tag' => 'nullable|string|default:latest',
            'ports' => 'nullable|array',
            'environment' => 'nullable|array',
            'volumes' => 'nullable|array',
            'restart_policy' => 'nullable|string|in:no,always,unless-stopped,on-failure',
        ]);

        // Create YAML definition
        $definition = [
            'slug' => $validated['slug'],
            'name' => $validated['name'],
            'description' => $validated['description'] ?? '',
            'category' => $validated['category'],
            'image' => $validated['image'],
            'registry' => $validated['registry'] ?? null,
            'tag' => $validated['tag'] ?? 'latest',
            'ports' => $validated['ports'] ?? [],
            'environment' => $validated['environment'] ?? [],
            'volumes' => $validated['volumes'] ?? [],
            'restart_policy' => $validated['restart_policy'] ?? 'unless-stopped',
        ];

        // Save to file
        $this->configService->saveDefinition($definition);

        // Sync to database
        $this->configService->syncToDatabase();

        return redirect()->route('admin.library.index')
            ->with('success', 'App definition created successfully');
    }

    public function show(AppDefinition $library)
    {
        return view('admin.library.show', ['definition' => $library]);
    }

    public function edit(AppDefinition $library)
    {
        return view('admin.library.edit', ['definition' => $library]);
    }

    public function update(Request $request, AppDefinition $library)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category' => 'required|string',
            'image' => 'required|string',
            'tag' => 'nullable|string',
            'ports' => 'nullable|array',
            'environment' => 'nullable|array',
            'volumes' => 'nullable|array',
        ]);

        $library->update($validated);

        return redirect()->route('admin.library.index')
            ->with('success', 'App definition updated');
    }

    public function destroy(AppDefinition $library)
    {
        $this->configService->deleteDefinition($library->slug);
        $library->delete();

        return redirect()->route('admin.library.index')
            ->with('success', 'App definition deleted');
    }

    public function sync()
    {
        $count = $this->configService->syncToDatabase();

        return redirect()->route('admin.library.index')
            ->with('success', "Synced {$count} app definitions");
    }
}
