<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Container;
use App\Models\AppDefinition;
use App\Services\DockerService;
use App\Services\NginxService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class ContainerController extends Controller
{
    public function __construct(
        protected DockerService $docker,
        protected NginxService $nginx
    ) {}

    public function index()
    {
        $dockerContainers = $this->docker->listContainers(true);
        $managedContainers = Container::all();

        return view('admin.containers.index', compact('dockerContainers', 'managedContainers'));
    }

    public function create()
    {
        $appDefinitions = AppDefinition::all();
        return view('admin.containers.create', compact('appDefinitions'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|regex:/^[a-z0-9-]+$/|unique:containers',
            'app_definition_id' => 'required|exists:app_definitions,id',
            'domain' => 'nullable|string|unique:containers',
            'ssl_enabled' => 'boolean',
            'auto_start' => 'boolean',
            'custom_ports' => 'nullable|array',
            'custom_environment' => 'nullable|array',
        ]);

        $appDef = AppDefinition::findOrFail($validated['app_definition_id']);

        // Determine ports
        $ports = $validated['custom_ports'] ?? $appDef->getDefaultPorts();

        // If auto-assign ports, find available ports
        if (empty($ports)) {
            $defaultContainerPort = 80;
            $availablePort = $this->docker->findAvailablePort();
            if ($availablePort) {
                $ports = [(string) $availablePort => (string) $defaultContainerPort];
            }
        }

        // Merge environment variables
        $environment = array_merge(
            $appDef->environment_defaults ?? [],
            $validated['custom_environment'] ?? []
        );

        // Create container config
        $containerConfig = [
            'name' => $validated['name'],
            'image' => $appDef->getFullImageName(),
            'network' => config('panel.docker.network'),
            'restart' => $appDef->restart_policy ?? 'unless-stopped',
            'environment' => $environment,
            'ports' => $ports,
            'volumes' => $this->buildVolumeMappings($appDef->volumes ?? [], $validated['name']),
            'labels' => array_merge(
                $appDef->labels ?? [],
                [
                    'managed-by' => config('panel.base_name') . '-panel',
                    'app-definition' => $appDef->slug,
                ]
            ),
        ];

        // Pull image first
        $this->docker->pullImage($containerConfig['image']);

        // Create Docker container
        $dockerContainerId = $this->docker->runContainer($containerConfig);

        if (!$dockerContainerId) {
            return back()->with('error', 'Failed to create container');
        }

        // Save to database
        $container = Container::create([
            'name' => $validated['name'],
            'app_definition_id' => $appDef->id,
            'docker_container_id' => $dockerContainerId,
            'status' => 'running',
            'domain' => $validated['domain'] ?? null,
            'ssl_enabled' => $validated['ssl_enabled'] ?? false,
            'port_mappings' => $ports,
            'environment' => $environment,
            'volumes' => $appDef->volumes ?? [],
            'auto_start' => $validated['auto_start'] ?? true,
        ]);

        // Configure Nginx if domain is set
        if (!empty($validated['domain'])) {
            // Request SSL certificate if enabled
            if ($validated['ssl_enabled'] ?? false) {
                $this->nginx->requestCertificate(
                    $validated['domain'],
                    config('panel.ssl.email')
                );
            }

            // Write Nginx config
            $this->nginx->writeConfig($container);
            $this->nginx->reload();
        }

        return redirect()->route('admin.containers.index')
            ->with('success', "Container '{$container->name}' created successfully");
    }

    public function show(Container $container)
    {
        $logs = '';
        $stats = null;

        if ($container->docker_container_id) {
            $logs = $this->docker->getLogs($container->docker_container_id, 500);
            $stats = $this->docker->getStats($container->docker_container_id);
        }

        $inspect = $container->docker_container_id 
            ? $this->docker->inspectContainer($container->docker_container_id) 
            : null;

        return view('admin.containers.show', compact('container', 'logs', 'stats', 'inspect'));
    }

    public function start(Container $container)
    {
        if ($container->docker_container_id) {
            $this->docker->startContainer($container->docker_container_id);
            $container->update(['status' => 'running']);
        }

        return back()->with('success', 'Container started');
    }

    public function stop(Container $container)
    {
        if ($container->docker_container_id) {
            $this->docker->stopContainer($container->docker_container_id);
            $container->update(['status' => 'exited']);
        }

        return back()->with('success', 'Container stopped');
    }

    public function restart(Container $container)
    {
        if ($container->docker_container_id) {
            $this->docker->restartContainer($container->docker_container_id);
            $container->update(['status' => 'running']);
        }

        return back()->with('success', 'Container restarted');
    }

    public function destroy(Container $container)
    {
        if ($container->docker_container_id) {
            // Remove container from Docker
            $this->docker->removeContainer($container->docker_container_id, true);
        }

        // Remove Nginx config
        $this->nginx->removeConfig($container);
        $this->nginx->reload();

        // Remove from database
        $container->delete();

        return redirect()->route('admin.containers.index')
            ->with('success', 'Container removed');
    }

    public function logs(Container $container)
    {
        $logs = '';
        if ($container->docker_container_id) {
            $logs = $this->docker->getLogs($container->docker_container_id, 1000);
        }

        return response()->json(['logs' => $logs]);
    }

    protected function buildVolumeMappings(array $volumeDefs, string $containerName): array
    {
        $mappings = [];
        $dataPath = config('panel.paths.data');

        foreach ($volumeDefs as $hostPath => $containerPath) {
            // Replace placeholders
            $hostPath = str_replace('{data_path}', $dataPath, $hostPath);
            $hostPath = str_replace('{container_name}', $containerName, $hostPath);

            // Create host directory if it doesn't exist
            if (!str_starts_with($hostPath, '/var/run/')) {
                if (!is_dir($hostPath)) {
                    mkdir($hostPath, 0755, true);
                }
            }

            $mappings[$hostPath] = $containerPath;
        }

        return $mappings;
    }
}
