<?php

namespace App\Console\Commands;

use App\Models\User;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class CreateAdminUser extends Command
{
    protected $signature = 'admin:create
                            {--username= : The username for the admin user}
                            {--email= : The email for the admin user}
                            {--password= : The password for the admin user}';

    protected $description = 'Create a new admin user';

    public function handle(): int
    {
        $username = $this->option('username') ?: $this->ask('Enter username');
        $email = $this->option('email') ?: $this->ask('Enter email address');
        $password = $this->option('password');

        if (!$password) {
            $password = $this->secret('Enter password');
            $confirmPassword = $this->secret('Confirm password');

            if ($password !== $confirmPassword) {
                $this->error('Passwords do not match!');
                return Command::FAILURE;
            }
        }

        // Validate input
        $validator = Validator::make([
            'username' => $username,
            'email' => $email,
            'password' => $password,
        ], [
            'username' => 'required|string|max:255|unique:users',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8',
        ]);

        if ($validator->fails()) {
            foreach ($validator->errors()->all() as $error) {
                $this->error($error);
            }
            return Command::FAILURE;
        }

        // Create user
        $user = User::create([
            'username' => $username,
            'email' => $email,
            'password' => Hash::make($password),
            'is_admin' => true,
        ]);

        $this->info("Admin user '{$user->username}' created successfully!");

        return Command::SUCCESS;
    }
}
