<?php

namespace App\Console\Commands;

use App\Models\Container;
use App\Services\DockerService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class ContainerHealthCheck extends Command
{
    protected $signature = 'containers:health-check';
    protected $description = 'Check health status of all managed containers';

    public function handle(DockerService $docker): int
    {
        if (!$docker->isAvailable()) {
            $this->error('Docker is not available');
            return Command::FAILURE;
        }

        $containers = Container::all();
        $checked = 0;
        $issues = 0;

        foreach ($containers as $container) {
            if (empty($container->docker_container_id)) {
                continue;
            }

            $info = $docker->inspectContainer($container->docker_container_id);

            if (!$info) {
                // Container no longer exists in Docker
                $container->update(['status' => 'missing', 'docker_container_id' => null]);
                Log::warning("Container {$container->name} no longer exists in Docker");
                $issues++;
                continue;
            }

            $state = $info['State']['Status'] ?? 'unknown';
            $running = $info['State']['Running'] ?? false;

            if ($state !== $container->status) {
                $container->update(['status' => $state]);
                Log::info("Container {$container->name} status changed from {$container->status} to {$state}");
            }

            // Check if auto-start is enabled but container is not running
            if ($container->auto_start && !$running && $state !== 'restarting') {
                $this->warn("Auto-start container {$container->name} is not running. Attempting restart...");
                $docker->startContainer($container->docker_container_id);
                Log::info("Auto-started container {$container->name}");
            }

            $checked++;
        }

        $this->info("Health check completed. Checked: {$checked}, Issues: {$issues}");

        return Command::SUCCESS;
    }
}
